/*---------------------------------------------------------------------------*\

    FILE....: COMM.CPP
    TYPE....: C+ Functions
    AUTHOR..: David Rowe
    DATE....: 20/11/97

    Functions used for PC to DSP communications, which includes passing
    messages to and from the DSP, and transfering information to and from
    FIFOs in the DSP.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include "comm.h"
#include "coff.h"
#include "hip.h"
#include "timer.h"
#include "mess.h"
#include "dspfifo.h"
#include "wobbly.h"
#include "generic.h"
#include "vpbapi.h"
#include "pci.h"
#ifdef LINUX
#include "hostdsp.h"
#endif

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <string.h>

/*--------------------------------------------------------------------------*\

				DEFINES

\*--------------------------------------------------------------------------*/

#define	WAIT	5	// time out delay in seconds		

/*--------------------------------------------------------------------------*\

				CLASS

\*--------------------------------------------------------------------------*/

class CommData {

	static int	exists;		// asserted if Commdata object exists		
	ULONG		avpbreg;	// address of registry info in DSP	
	USHORT		lvpbreg;	// length of registry info in DSP	
	ULONG		a_asst;		// address of assert flag in DSP
	char		firm[MAX_STR];	// firmware file name

public:
	VPBRegister	*v;		// object that stores system config info
	Hip		*hip;		// object for talking to VPB hips
	USHORT		numvpb;		// number of VPB cards 			

	CommData();
	virtual ~CommData();
        void InitDefault(VPBREG *vr);
        void InitV12PCI(VPBREG *vr);
	int PutMessageVPB(USHORT board, word *mess);
	int GetMessageVPB(USHORT board, word *mess);
	void WaitForMessageVPB(USHORT board, word *mess, USHORT mtype, USHORT wait);
	virtual void CheckForAssert(USHORT board);
	void GetStringFromDSP(USHORT board, char s[], char name[]);
};

int CommData::exists = 0;

// Critical section for PutMessageVPBCriticalSect() to prevent API
// functions and timer callback function writing to DSP message Q
// at the same time

GENERIC_CRITICAL_SECTION	PutMessageSect;

/*-------------------------------------------------------------------------*\

			COMM MEMBER FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::Comm
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Initialises the VPBs to the desired configuration, and boots the
	DSP programs in each VPB.  After this function is called,
	communications I/O can proceed between the VPB and the PC.

\*--------------------------------------------------------------------------*/

Comm::Comm()
{
	d = new CommData();
	if (d == NULL)
		throw Wobbly(COMM_CANT_ALLOCATE_MEMORY);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::~Comm
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Closes down the comm link to the VPBs.

\*--------------------------------------------------------------------------*/

Comm::~Comm()
{
	delete d;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::PutMessageVPB
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Sends a message to the DSPs message queue.  Returns OK if successful,
	or COMM_FULL if DSP message queue full.  Length is in first word
	of mess[].

\*--------------------------------------------------------------------------*/

int Comm::PutMessageVPB(USHORT board, word *mess)
//  USHORT  board;		VPB board number
//  word    *mess;		ptr to message				
{
    return(d->PutMessageVPB(board, mess));
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::GetMessageVPB
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Gets a message from the DSPs message queue.  Length of message is
	stored in first word of message.  Returns OK if message read,
	else COMM_EMPTY if no messages on queue.

\*--------------------------------------------------------------------------*/

int Comm::GetMessageVPB(USHORT board, word *mess)
//  USHORT  board;		VPB board number
//  word    *mess;		ptr to message				
{
	return(d->GetMessageVPB(board, mess));
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::WaitForMessageVPB
	AUTHOR...: David Rowe
	DATE.....: 28/11/97

	Keeps checking the DSPs message for a message of a certain type,
	messages of other types are discarded.  If the desired message is
	not found after a specified time a time out wobbly is thrown.

	The caller is responsible for having enough storage in mess[] for
	the desired message.

\*--------------------------------------------------------------------------*/

void Comm::WaitForMessageVPB(USHORT board, word *mess, USHORT mtype, USHORT wait)
//  USHORT  board;		VPB board number
//  word    *mess;		ptr to message	
//  USHORT  mtype;		type of message to wait for
//  USHORT  wait;		timeout delay in seconds			
{
    d->WaitForMessageVPB(board, mess, mtype, wait);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::vpbreg
	AUTHOR...: David Rowe
	DATE.....: 28/11/97

	Nasty un-object oriented function to return a pointer to the 
	vpbreg structure for a given board.

\*--------------------------------------------------------------------------*/

VPBREG *Comm::vpbreg(USHORT board)
//	USHORT	board;		VPB board number
{
	return(&d->v->reg[board]);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::numboards
	AUTHOR...: David Rowe
	DATE.....: 4/3/98

	Returns the number of boards controlled by Com object.

\*--------------------------------------------------------------------------*/

USHORT Comm::numboards()
{
	return(d->numvpb);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::hip
	AUTHOR...: David Rowe
	DATE.....: 28/11/97

	Nasty un-object oriented function to return a pointer to the 
	hip strucuture.

\*--------------------------------------------------------------------------*/

Hip *Comm::hip(USHORT board)
//	USHORT	board;		VPB board number
{
	return(d->hip);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::CheckForAssert
	AUTHOR...: David Rowe
	DATE.....: 1/12/97

	Debug function that checks the DSP to see if an assert has
	occurred, which hangs the DSP.  If an assert occurs this
	function throws a Wobbly.

\*--------------------------------------------------------------------------*/

void Comm::CheckForAssert(USHORT board)
//	USHORT	board;		VPB board number
{
    d->CheckForAssert(board);
}

/*-------------------------------------------------------------------------*\

			COMMDATA MEMBER FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::CommData
	AUTHOR...: David Rowe
	DATE.....: 4/2/98

	Initialises the VPBs to the desired configuration, and boots the
	DSP programs in each VPB.  After this function is called,
	communications I/O can proceed between the VPB and the PC.

\*--------------------------------------------------------------------------*/

CommData::CommData()
{
	VPBREG	 *vr;
	
	// bomb out if CommData already exists

	if (exists)
		throw Wobbly(COMM_ALREADY_INITIALISED);
	exists = 1;

	// Read registry and init HIP

	v = new VPBRegister(&numvpb);
	if (v == NULL)
		throw Wobbly(COMM_CANT_ALLOCATE_MEMORY);
	vr = v->reg;
	
	switch (vr->model) {
	case VPB_V12PCI:
	        InitV12PCI(vr);
	break;
	default:
		InitDefault(vr);
	}

	GenericInitializeCriticalSection(&PutMessageSect);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::InitDefault
	AUTHOR...: David Rowe
	DATE.....: 13/5/02

	Default init function used for DSP-based cards.

\*--------------------------------------------------------------------------*/

void CommData::InitDefault(VPBREG *vr) {
	word     data;			// copy of data variable from DSP mem	
	Timer    timer;			// timer object			
	USHORT   time_out;		// == TIME_OUT if time out occurred	
	ULONG	 a_model;		// address of model in firmware
	word	 model;			// model number read from firmware
	int      i;	
	USHORT   eebuf[128];		// buffer for card detail extraction
        char    hex[17]={"0123456789abcdef"}; 	// translation table.
	
	hip = new Hip(vr->ddmodel);
	if (hip == NULL)
		throw Wobbly(COMM_CANT_ALLOCATE_MEMORY);

	// load a few constants 

	coff_get_address(vr->firm, "_model", &a_model);
	coff_get_address(vr->firm, "_asst", &a_asst);
	strcpy(firm, vr->firm);

	// Boot each VPB ----------------------------------------------------

	dspfifo_open(vr->szRelayBuf);
	coff_get_address(vr->firm, "_vpbreg", &avpbreg);
	lvpbreg = &vr[0].base - &vr[0].data;
	for(i=0; i<numvpb; i++) {
	
		// init the HIP for the current VPB 

		hip->InitVpb(vr[i].base);

		// download firmware and start DSP

		mprintf("About to load firmware...\n");
		coff_load_dsp_firmware(hip, i, vr->firm);
		mprintf("About to run...\n");
		GenericSleep(10);
		hip->DspRun(i);
		GenericSleep(10);
		mprintf("DSP running...\n");

		//#define DSPRUN
		#ifdef DSPRUN
		mprintf("DSP running...\n");
		while(1);
		#endif

		// download config structure and start DSP
		
		vr[i].data = 0;
		hip->WriteDspSram(i, (USHORT)avpbreg, lvpbreg, (word*)&vr[i]);

		// now assert "data" to signal DSP to start its config

		vr[i].data = 1;
		hip->WriteDspSram(i, (USHORT)avpbreg, 1, (word*)&vr[i]);

		// DSP then fills in certain parameters in the VPBREG
		// structure which we can upload, such as the location
		// of the message fifos in DSP memory.
		// when DSP has finished it's init, it writes a 0 to "data" 

		timer.timer_start();
		do {
			hip->ReadDspSram(i, (USHORT)avpbreg, 1, &data);
			timer.timer_check_time_out(WAIT, &time_out);
		} while((data != 0) && (time_out == OK));

		if (time_out == TIME_OUT) {
			// first check for DSP assert

			CheckForAssert(i);

			throw Wobbly(COMM_TIME_OUT_CONFIGURING_DSP);
		}

		// OK, up load reg structure to fill in DSP init info	

		hip->ReadDspSram(i, (USHORT)avpbreg, lvpbreg, (word*)&vr[i]);

		// check model number in registry matches firmware
		// we do this here to make sure DSP has enough time to set 
		// "model" static

		hip->ReadDspSram(i, (USHORT)a_model, 1, (word*)&model);
		if (model != vr[i].model) {
			printf("model = %d  vr[i].model = %d\n",model, 
			       vr[i].model);
			throw Wobbly(COMM_FIRMWARE_MODEL_NUMBER);
		}

		// now up perform PC side of message DSP FIFO initialisation

		vr[i].dnmess = new DspFifo(hip, i, vr->a_dnmess, DSPFIFO_DOWN,
					   DSPFIFO_NORMAL);
		vr[i].upmess = new DspFifo(hip, i, vr->a_upmess, DSPFIFO_UP, 
					   DSPFIFO_NORMAL);

		mprintf("DSP [%02d] Message FIFOs booted OK\n",i);

		// now read card version data, add to struct table
	
		#ifndef FREEBSD

		// only supported under linux for now

		hip->EeRead(i,0,64,eebuf);	
		
	        vr[i].mdate[0]=hex[((eebuf[50]>>12)& 0x0f)];    //day
	        vr[i].mdate[1]=hex[((eebuf[50]>>8)& 0x0f)];
	        vr[i].mdate[2]='/';
	        vr[i].mdate[3]=hex[((eebuf[50]>>4)& 0x0f)];     //month
	        vr[i].mdate[4]=hex[(eebuf[50] & 0x0f)];
	        vr[i].mdate[5]='/';
	        vr[i].mdate[6]=hex[((eebuf[51]>>12)& 0x0f)];    //year
	        vr[i].mdate[7]=hex[((eebuf[51]>>8)& 0x0f)];
	        vr[i].mdate[8]=hex[((eebuf[51]>>4)& 0x0f)];
	        vr[i].mdate[9]=hex[(eebuf[51] & 0x0f)];
	        vr[i].mdate[10]=0;
		// Decode Rev
                vr[i].revision[0]=hex[((eebuf[52]>>12)& 0x0f)];    //card
                vr[i].revision[1]=hex[((eebuf[52]>>8)& 0x0f)];
                vr[i].revision[2]='.';
                vr[i].revision[3]=hex[((eebuf[52]>>4)& 0x0f)];     //Rev.
                vr[i].revision[4]=hex[(eebuf[52] & 0x0f)];
                vr[i].revision[5]=0;
                // Decode SN:
                vr[i].serial_n[0]=hex[((eebuf[54]>>12)& 0x0f)];    //SN:high
                vr[i].serial_n[1]=hex[((eebuf[54]>>8)& 0x0f)];
                vr[i].serial_n[2]=hex[((eebuf[54]>>4)& 0x0f)];
                vr[i].serial_n[3]=hex[(eebuf[54] & 0x0f)];
                vr[i].serial_n[4]=hex[((eebuf[55]>>12)& 0x0f)];    //SN:low
                vr[i].serial_n[5]=hex[((eebuf[55]>>8)& 0x0f)];
                vr[i].serial_n[6]=hex[((eebuf[55]>>4)& 0x0f)];
                vr[i].serial_n[7]=hex[(eebuf[55] & 0x0f)];
                vr[i].serial_n[8]=0;
		#else
		memset(eebuf, 0, sizeof(eebuf));
		#endif
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::InitV12PCI
	AUTHOR...: David Rowe
	DATE.....: 13/5/02

	Init function used for host-based cards.

\*--------------------------------------------------------------------------*/

void CommData::InitV12PCI(VPBREG *vr) {
	int i;
	
	// Boot each V12PCI --------------------------------------------------

	dspfifo_open(vr->szRelayBuf);
	for(i=0; i<numvpb; i++) {
			
		// init message DSP FIFOs

		vr[i].dnmess = new DspFifo(DSPFIFO_DOWN, vr->szmess, 
					   DSPFIFO_NORMAL);
		vr[i].upmess = new DspFifo(DSPFIFO_UP, vr->szmess, 
					   DSPFIFO_NORMAL);

		mprintf("DSP [%02d] Message FIFOs booted OK\n",i);
	
	}

	// Start "DSP" (simulated by thread)
	#ifdef LINUX
	printf("Starting host DSP!\n");
	HostDSPOpen(numvpb, vr);
	#endif
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::~CommData
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Closes down the comm link to the VPBs.

\*--------------------------------------------------------------------------*/

CommData::~CommData()
{
	int      i;

	// free storage used by message FIFOs 

	if (v->reg->model != VPB_V12PCI) {
		// check memory unmodified
	
		for(i=0; i<numvpb; i++)
			coff_check_dsp_firmware(hip, i, v->reg->firm);
		delete hip;
	}
	else {
		#ifdef LINUX
		HostDSPClose();
		#endif
	}

	for(i=0; i<numvpb; i++) {
		delete v->reg[i].dnmess;
		delete v->reg[i].upmess;
	}

	delete v;

	GenericDeleteCriticalSection(&PutMessageSect);
	dspfifo_close();

	exists = 0;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::PutMessageVPB
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Sends a message to the DSPs message queue.  Returns OK if successful,
	or COMM_FULL if DSP message queue full.  Length is in first word
	of mess[].

\*--------------------------------------------------------------------------*/

int CommData::PutMessageVPB(USHORT board, word *mess)
//	USHORT	board;		VPB board number
//	word    *mess;		ptr to message				
{
	int ret;

	GenericEnterCriticalSection(&PutMessageSect);

        do {
		ret = v->reg[board].dnmess->Write(mess, mess[0]);
		if (ret == DSPFIFO_FULL) {
			//mprintf("board %d dn mess fifo full\n",board);
		  GenericSleep(5);	// dont hammer DSP
		}
	} while (ret == DSPFIFO_FULL);

	CheckForAssert(board);
	GenericLeaveCriticalSection(&PutMessageSect);
	//if (ret == DSPFIFO_FULL)
	//	throw Wobbly(COMM_DOWN_MESSAGE_Q_FULL);
	return(ret);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::GetMessageVPB
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Gets a message from the DSPs message queue.  Length of message is
	stored in first word of message.  Returns OK if message read,
	else COMM_EMPTY if no messages on queue.

\*--------------------------------------------------------------------------*/

int CommData::GetMessageVPB(USHORT board, word *mess)
//	USHORT	board;		VPB board number
//	word    *mess;		ptr to message				
{
	int     ret;
	VPBREG	*vr = &v->reg[board];

	assert(mess != NULL);

	// get length of message (first word in message) 

	ret = vr->upmess->Read(mess, 1);
	CheckForAssert(board);
	if (ret != OK)
		return(COMM_EMPTY);

	// get rest of message

	ret = vr->upmess->Read(&mess[1], mess[0]-1);
	CheckForAssert(board);
    
	// this should never happen unless DSP crashed

	if (ret != OK) {
		throw Wobbly(COMM_DSP_MESSAGE_CORRUPT);
	}
    
	return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::WaitForMessageVPB
	AUTHOR...: David Rowe
	DATE.....: 28/11/97

	Keeps checking the DSPs message for a message of a certain type,
	messages of other types are discarded.  If the desired message is
	not found after a specified time a time out wobbly is thrown.

	The caller is responsible for having enough storage in mess[] for
	the desired message.

\*--------------------------------------------------------------------------*/

void CommData::WaitForMessageVPB(USHORT board, word *mess, USHORT mtype, USHORT wait)
//  USHORT	board;		VPB board number
//  word    *mess;		ptr to message	
//  USHORT  mtype;		type of message to wait for
//  USHORT  wait;		timeout delay in seconds			
{
	Timer	timer;
	word	m[COMM_MAX_MESS];	// big enough to hold any mess
	USHORT	found;
	USHORT  ret;
	USHORT  time_out;

	assert(board < MAX_VPB);
	assert(mess != NULL);
	assert(wait != 0);

	found = 0;
	timer.timer_start();
	do {
		ret = GetMessageVPB(board, m);
		if (ret == OK) {
			if (m[1] == mtype) {
				memcpy(mess, m, sizeof(word)*m[0]);
				found = 1;
			}
		}
		timer.timer_check_time_out(wait, &time_out);
	} while((found == 0) && (time_out == OK));

	if (time_out == TIME_OUT) {
		// first check for asserts in DSP

		CheckForAssert(board);

		throw Wobbly(COMM_TIME_OUT_WAITING_FOR_MESS);
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::CheckForAssert
	AUTHOR...: David Rowe
	DATE.....: 1/12/97

	Debug function that checks the DSP to see if an assert has
	occurred, which hangs the DSP.  If an assert occurs this
	function throws a Wobbly.

\*--------------------------------------------------------------------------*/

void CommData::CheckForAssert(USHORT board)
//	USHORT	board;		VPB board number
{
	word	asst;		// asst value read from DSP
	char	gmsg[MAX_STR];	// assert message from DSP
	char	gcnd[MAX_STR];	// assert condition from DSP
	char	gfile[MAX_STR];	// assert file from DSP
	word	gline;		// assert line number from DSP
	ULONG	a_gline;	// address of gline in DSP
   
	if (v->reg->model == VPB_V12PCI)
		return;

	assert(board < MAX_VPB);
	hip->ReadDspSram(board, (USHORT)a_asst, 1, &asst);

	if (asst) {
	    
		// if asserted get assert params from DSP

		GetStringFromDSP(board, gmsg, "_gmsg");
		GetStringFromDSP(board, gcnd, "_gcnd");
		GetStringFromDSP(board, gfile, "_gfile");

  		coff_get_address(firm, "_gline", &a_gline);
		hip->ReadDspSram(board, (USHORT)a_gline, 1, &gline);
	
		// output string length must be < MAX_STR

		assert((strlen(gfile) + 5) < MAX_STR);

		// print debug data
#ifdef TMP
  		coff_get_address(firm, "_ad1", &a_data);
		hip->ReadDspSram(board, (USHORT)a_data, 1, &data);
		printf("\nassert data1 = %d [0x%x]\n",data,data);
  	      
		coff_get_address(firm, "_ad2", &a_data);
		hip->ReadDspSram(board, (USHORT)a_data, 1, &data);
		printf("assert data2 = %d [0x%x]\n",data,data);
  		coff_get_address(firm, "_ad3", &a_data);
		hip->ReadDspSram(board, (USHORT)a_data, 1, &data);
		printf("assert data3 = %d [0x%x]\n",data,data);
#endif		
		// throw wobbly (note only using file and line num)

		#ifdef Wobbly
		#undef Wobbly			// turn off macro
		throw Wobbly(COMM_DSP_ASSERT, gfile, gline);
#include "wobbly.h"	// turn macro back on
		#else
		throw Wobbly(COMM_DSP_ASSERT, gfile, gline);
		#endif

	}
    
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::GetStringFromDSP
	AUTHOR...: David Rowe
	DATE.....: 1/12/97

	Obtains a text string from the DSP, the global name passed to
	this function is assumed to point to the string required.

\*--------------------------------------------------------------------------*/

void CommData::GetStringFromDSP(USHORT board, char s[], char name[])
//  USHORT  board;		VPB board number
//  char    s[];		string uploaded from DSP
//  char    name[];		name of ptr to string in DSP program
{
	ULONG   a_sword;	// address of string in DSP
	word    sword[MAX_STR];	// string in DSP 16 bit word per char format
	word	addr;		// address of string
	int	i;

	// validate arguments

	assert(board < MAX_VPB);
	assert(s != NULL);
	assert(name != NULL);

	// get address of ptr, load ptr and hence string

	coff_get_address(firm, name, &a_sword);
	hip->ReadDspSram(board, (USHORT)a_sword, 1, &addr);
	hip->ReadDspSram(board, addr, MAX_STR, sword);

	// convert to string made of 8 bit chars

	i = 0;
	do {
		s[i] = (char)sword[i];
		i++;
	} while ( s[i-1] && (i < (MAX_STR-1)) );

	s[i] = 0;
}

